<?php

/**
 * The admin-specific functionality of the plugin.
 *
 * @link       http://www.castorstudio.com
 * @since      1.0.0
 *
 * @package    Uwac
 * @subpackage Uwac/admin
 */

class Uwac_Admin {

	/**
	 * The ID of this plugin.
	 *
	 * @since    1.0.0
	 * @access   private
	 * @var      string    $plugin_name    The ID of this plugin.
	 */
	private $plugin_name;

	/**
	 * The version of this plugin.
	 *
	 * @since    1.0.0
	 * @access   private
	 * @var      string    $version    The current version of this plugin.
	 */
	private $version;

	/**
	 * Initialize the class and set its properties.
	 *
	 * @since    1.0.0
	 * @param      string    $plugin_name       The name of this plugin.
	 * @param      string    $version    The version of this plugin.
	 */
	public function __construct( $plugin_name, $version ) {

		$this->plugin_name = $plugin_name;
		$this->version = $version;

	}

	/**
	 * Register the stylesheets for the admin area.
	 *
	 * @since    1.0.0
	 */
	public function enqueue_styles() {
		wp_enqueue_style( $this->plugin_name, plugin_dir_url( __FILE__ ) . 'css/uwac-admin.css', array(), $this->version, 'all' );
		wp_enqueue_style( $this->plugin_name.'_public', CS_UWAC_PLUGIN_URI . '/public/css/uwac-public.css', array(), $this->version, 'all' );

		// AJAX CALL: All Available Themes (for all customizable sections)
		wp_enqueue_style( $this->plugin_name . '_dynamic-themes',admin_url('admin-ajax.php').'?action=uwac_dynamic_themes', array($this->plugin_name), $this->version, 'all');
	}

	/**
	 * Register the JavaScript for the admin area.
	 *
	 * @since    1.0.0
	 */
	public function enqueue_scripts() {
		wp_enqueue_script( $this->plugin_name, plugin_dir_url( __FILE__ ) . 'js/uwac-admin.js', array( 'jquery' ), $this->version, false );
		wp_enqueue_script( $this->plugin_name.'_public', CS_UWAC_PLUGIN_URI . '/public/js/uwac-public.js', array( 'jquery' ), $this->version, false );
	}




	
	function load_plugin_config(){
		require_once( 'config/uwac-config.php'  );
	}


	/**
	 * Get Available Themes for Settings Framework
	 *
	 * @since    1.0.0
	 */
	public static function __get_themes($preview = true){
		$active_themes = Uwac::get_themes()->get_themes();
		$output_field_options = array();

		if (isset($active_themes['widget_themes'])){
			foreach ($active_themes['widget_themes'] as $theme){
				if ($theme->type == 'dynamic'){
					$output_field_options[$theme->raw_name] = array(
						'image'	=> $theme->uri .'/preview.png',
						'name'	=> $theme->human_name,
					);
				}
			}
		}
		return $output_field_options;
	}
	public static function get_themes($theme_family){
		$active_themes = Uwac::get_themes()->get_themes();
		$output_field_options = array();

		if (isset($active_themes[$theme_family])){
			foreach ($active_themes[$theme_family] as $theme){
				if ($theme->type == 'dynamic'){
					$output_field_options[$theme->raw_name] = array(
						'image'	=> $theme->uri .'/preview.png',
						'name'	=> $theme->human_name,
					);
				}
			}
		}
		return $output_field_options;
	}


	/**
	 * Get Available Admin Themes Settings for Settings Framework
	 *
	 * @since    1.0.0
	 */
	public static function __get_themes_settings(){
		$active_themes = Uwac::get_themes()->get_themes();
		$settings = array();

		if (isset($active_themes['themes'])){
			foreach ($active_themes['themes'] as $theme){
				if ($theme->type == 'dynamic'){
					$settings[] = Uwac::get_theme($theme->object_name)->get_settings();
				}
			}
		}
		return $settings;
	}
	public static function get_themes_settings($theme_family){
		$active_themes = Uwac::get_themes()->get_themes();
		$settings = array();

		if (isset($active_themes[$theme_family])){
			foreach ($active_themes[$theme_family] as $theme){
				if ($theme->type == 'dynamic'){
					$settings[] = $theme->instance->get_settings();
				}
			}
		}
		return $settings;
	}


	/**
	 * Restrict Access to admin settings
	 * 
	 * @since 1.1.0
	 * @version 1.0.0
	 *
	 * @return void
	 */
	function show_admin_menu(){
		$accessrights 	= cs_uwac_get_settings('accessrights');

		if ($accessrights === 'all' || !$accessrights){
			return true;
		} else if ($accessrights == 'superadmin'){
			if (is_super_admin()){
				return true;
			} else { 
				return false;
			}
		}
	}


	/**
	 * Floating Widget View
	 */
	function floating_widget_view(){
		Uwac_Common::floating_widget_view();
	}


	/**
	 * Plugin row action links
	 */
	function plugin_row_action_links($actions,$file){
		$ipido_basename = 'uwac/uwac.php';
		if ($ipido_basename != $file){ return $actions; }

		$settings = array('settings' => '<a href="admin.php?page=cs-uwac">' . __('Settings','uwac') . '</a>');
		$site_link = array('support' => '<a href="' . CS_UWAC_PLUGIN_URL . '/support/" target="_blank">'. __('Support','uwac') .'</a>');

		$actions = array_merge($settings, $actions);
		$actions = array_merge($site_link, $actions);

		return $actions;
	}

	/**
	 * Plugin row meta links
	 */
	function plugin_row_meta_links( $input, $file ) {
		$ipido_basename = 'uwac/uwac.php';
		if ($ipido_basename != $file){ return $input; }

		$links = array(
			'<a href="' . admin_url( 'admin.php?page=cs-uwac' ) . '">' . __( 'Getting Started','uwac' ) . '</a>',
			'<a href="' . CS_UWAC_PLUGIN_URL . '/docs/" target="_blank">' . __( 'Documentation','uwac' ) . '</a>',
		);

		$output = array_merge( $input, $links );

		return $output;
	}


	/**
	 * Register the 'Admin Menu Manager Page' as a submenu page
	 * 
	 * @since 1.0.0
	 */
	function register_admin_pages($framework_unique){
		$page_hook = add_submenu_page('cs-uwac', 'UWAC About', __( 'About the Plugin','uwac'), 'manage_options', 'cs-uwac-about', 'cs_uwac_welcome_page' );
		add_action("load-{$page_hook}",array(&$this,'cs_uwac_register_about_plugin_page'));
	}

	function cs_uwac_register_about_plugin_page(){
		wp_register_style( $this->plugin_name .'_about', plugin_dir_url( __FILE__ ) . 'css/uwac-page-dashboard.css' );
		wp_enqueue_style( $this->plugin_name .'_about' );
	}
}